/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { JSONPatchDocument } from '@dcx/common-types';
import { createPatchDocumentBuilder, generateUuid, pruneUndefined } from '@dcx/util';
import { AdobeXMPUtilContext, _getPatchDocumentFromContext } from './context';
import { DEFAULT_CREATOR_TOOL } from './private';

export function appendHistoryEvent(
    this: AdobeXMPUtilContext | void | unknown,
    action: string,
    creatorTool: string = DEFAULT_CREATOR_TOOL,
    when: string = new Date().toISOString(),
    instanceId: string = generateUuid(),
): JSONPatchDocument {
    const doc = _getPatchDocumentFromContext(this);
    const event = _createHistoryEvent(instanceId, action, creatorTool, when);
    return createPatchDocumentBuilder(doc)
        .add('/xmpMM:History/@list/-', event)
        .replace('/xmpMM:InstanceID', instanceId)
        .replace('/xmp:ModifyDate', when)
        .replace('/xmp:MetadataDate', when).operations;
}

export function updateLastHistoryEvent(
    this: AdobeXMPUtilContext | void | unknown,
    action?: string,
    creatorTool?: string,
    when?: string,
): JSONPatchDocument {
    const doc = _getPatchDocumentFromContext(this);
    if (action == null && creatorTool == null && when == null) {
        return doc;
    }

    const builder = createPatchDocumentBuilder(doc);
    let event = _createHistoryEvent(generateUuid(), action as string, creatorTool as string, when as string);
    event = pruneUndefined(event);

    for (const k in event) {
        builder.replace(`/xmpMM:History/@list/-/${k}`, event[k]);
    }
    return builder.operations;
}

export function makeDerivedWithAction(
    this: AdobeXMPUtilContext | void | unknown,
    creatorTool: string = DEFAULT_CREATOR_TOOL,
    action: string,
    newDocId: string = generateUuid(),
    when: string = new Date().toISOString(),
): JSONPatchDocument {
    const doc = _getPatchDocumentFromContext(this);

    const builder = createPatchDocumentBuilder(doc)
        .add('/@context/stRef', 'http://ns.adobe.com/xap/1.0/sType/ResourceRef#')
        .add('/@context/xmpMM', 'http://ns.adobe.com/xap/1.0/mm/')
        .add('/xmpMM:DerivedFrom', {})
        .copy('/xmpMM:DocumentID', '/xmpMM:DerivedFrom/stRef:documentID')
        .copy('/xmp:ModifyDate', '/xmpMM:DerivedFrom/stRef:lastModifyDate')
        .copy('/xmpMM:InstanceID', '/xmpMM:DerivedFrom/stRef:instanceID')
        .copy('/xmpMM:OriginalDocumentID', '/xmpMM:DerivedFrom/stRef:originalDocumentID')
        .replace('/xmpMM:InstanceID', newDocId)
        .replace('/xmpMM:DocumentID', newDocId);
    return appendHistoryEvent.call({ patchDocument: builder.operations }, action, creatorTool, when, newDocId);
}

export function setProperty(
    this: AdobeXMPUtilContext | void | unknown,
    namespace: string,
    name: string,
    value: unknown,
): JSONPatchDocument {
    const doc = _getPatchDocumentFromContext(this);
    return createPatchDocumentBuilder(doc).add(`/${namespace}:${name}`, value).operations;
}

export function setCreatorTool(this: AdobeXMPUtilContext | void | unknown, creatorTool: string): JSONPatchDocument {
    return setProperty.call(this, 'xmp', 'CreatorTool', creatorTool);
}

export function deleteProperty(this: AdobeXMPUtilContext | void | unknown, namespace: string, name: string) {
    const doc = _getPatchDocumentFromContext(this);
    return createPatchDocumentBuilder(doc).remove(`/${namespace}:${name}`).operations;
}

function _createHistoryEvent(instanceId: string, action: string, creatorTool: string, when: string) {
    return {
        'stEvt:action': action,
        'stEvt:when': when,
        'stEvt:softwareAgent': creatorTool,
        'stEvt:instanceID': instanceId,
    };
}
