/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { JSONPatchDocument } from '@dcx/common-types';
import { createPatchDocumentBuilder, generateUuid } from '@dcx/util';
import { AdobeXMPUtilContext, _getPatchDocumentFromContext } from './context';
import { DEFAULT_CREATOR_TOOL } from './private';
import { makeHistoryEventXML } from './xml';

export function registerNamespace(this: AdobeXMPUtilContext | void, ns: string, shortName: string): JSONPatchDocument {
    const doc = _getPatchDocumentFromContext(this);
    return createPatchDocumentBuilder(doc).add(`/@context/${shortName}`, ns).operations;
}

export function initializeXMPJSON(
    creatorTool = DEFAULT_CREATOR_TOOL,
    docId: string = generateUuid(),
    when: string = new Date().toISOString(),
) {
    return {
        'xmp:ModifyDate': when,
        'xmp:MetadataDate': when,
        'xmp:CreatorTool': creatorTool,
        '@context': {
            xmpMM: 'http://ns.adobe.com/xap/1.0/mm/',
            stEvt: 'http://ns.adobe.com/xap/1.0/sType/ResourceEvent#',
            xmp: 'http://ns.adobe.com/xap/1.0/',
        },
        'xmpMM:DocumentID': `xmp.did:${docId}`,
        'xmpMM:History': {
            '@list': [
                {
                    'stEvt:softwareAgent': creatorTool,
                    'stEvt:action': 'created',
                    'stEvt:when': when,
                    'stEvt:instanceID': `xmp.iid:${docId}`,
                },
            ],
        },
        'xmp:CreateDate': when,
        '@id': '',
        'xmpMM:InstanceID': `xmp.iid:${docId}`,
        'xmpMM:OriginalDocumentID': `xmp.did:${docId}`,
    };
}

export function initializeXMPXML(
    creatorTool = DEFAULT_CREATOR_TOOL,
    docId: string = generateUuid(),
    when: string = new Date().toISOString(),
    historyEvents?: string[],
    derivedWithData?: string,
): string {
    if (!historyEvents) {
        historyEvents = [makeHistoryEventXML(creatorTool, 'created', docId, when)];
    }
    return `<x:xmpmeta xmlns:x="adobe:ns:meta/" x:xmptk="Adobe XMP Core 7.1.2">
        <rdf:RDF xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#">
            <rdf:Description rdf:about=""
            xmlns:xmp="http://ns.adobe.com/xap/1.0/"
            xmlns:xmpMM="http://ns.adobe.com/xap/1.0/mm/"
            ${derivedWithData ? 'xmlns:stRef="http://ns.adobe.com/xap/1.0/sType/ResourceRef#"' : ''}
            xmlns:stEvt="http://ns.adobe.com/xap/1.0/sType/ResourceEvent#"
            xmpMM:OriginalDocumentID="xmp.did:${docId}"
            xmp:ModifyDate="${when}"
            xmp:MetadataDate="${when}"
            xmp:CreatorTool="${creatorTool}"
            xmp:CreateDate="${when}"
            xmpMM:DocumentID="xmp.did:${docId}"
            xmpMM:InstanceID="xmp.iid:${docId}">
                <xmpMM:History>
                    <rdf:Seq>
                        ${historyEvents.join('\n')}
                    </rdf:Seq>
                </xmpMM:History>
                ${derivedWithData ? derivedWithData : ''}
            </rdf:Description>
        </rdf:RDF>
    </x:xmpmeta>`;
}
