import { generateUuid } from '@dcx/util';
import { initializeXMPXML } from './core';
import { DEFAULT_CREATOR_TOOL } from './private';
import { parseXML, XMLElement } from './XMLParser';

/**
 * Incoming XML may or may not include all fields
 * in the stub defined in core. If not, add missing fields.
 * Generates document/instance IDs if they don't exist.
 *
 * @note
 * This repair is not extensive. If a field exists with an invalid value,
 * it will not be repaired. Likewise for certain expected history events.
 */
export function repairXMPXML(
    xmp: string,
    creatorTool = DEFAULT_CREATOR_TOOL,
    docId: string = generateUuid(),
    when: string = new Date().toISOString(),
): string {
    const parser = parseXML(xmp);
    const xmpmeta = parser.getElementsByTagName('x:xmpmeta');
    if (xmpmeta.length < 1) {
        return initializeXMPXML(creatorTool, docId, when);
    }

    const rdf = xmpmeta[0].getElementsByTagName('rdf:RDF');
    if (rdf.length < 1) {
        return initializeXMPXML(creatorTool, docId, when);
    }

    const rdfDescription = rdf[0].getElementsByTagName('rdf:Description');

    if (rdfDescription.length < 1) {
        return initializeXMPXML(creatorTool, docId, when);
    }
    const desc = rdfDescription[0];

    repairElement(desc, 'xmp:ModifyDate', when);
    repairElement(desc, 'xmp:MetadataDate', when);
    repairElement(desc, 'xmp:CreatorTool', creatorTool);
    repairElement(desc, 'xmp:CreateDate', when);
    repairElement(desc, 'xmpMM:DocumentID', `xmp.did:${docId}`);
    repairElement(desc, 'xmpMM:InstanceID', `xmp.iid:${docId}`);
    repairElement(desc, 'xmpMM:OriginalDocumentID', `xmp.did:${docId}`);

    repairHistory(desc);

    return parser.toString();
}

function repairElement(elem: XMLElement, tagName: string, initVal: string) {
    const elems = elem.getElementsByTagName(tagName);
    const attr = elem.getAttribute(tagName);

    if (elems.length < 1 && attr == null) {
        elem.setAttribute(tagName, initVal);
    }
}

function repairHistory(description: XMLElement) {
    const tags = ['xmpMM:History', 'rdf:Seq'];
    let currNode = description;
    while (tags.length > 0) {
        const tag = tags.shift() as string;
        const elems = currNode.getElementsByTagName(tag);
        if (elems.length > 0) {
            currNode = elems[0];
        } else {
            currNode = currNode.appendChild(tag);
        }
    }
    // currNode is now history>seq

    // TODO:? Decide whether we want to add a created event,
    // even though we don't know when/how it was created.
}
