const { execSync } = require('child_process');
const path = require('path');
const fs = require('fs');
const os = require('os');

// Chrome versions to download
const CHROME_VERSION = '138.0.7204.49';

// Detect current platform
const currentPlatform = os.platform();
const isMac = currentPlatform === 'darwin';
const isWindows = currentPlatform === 'win32';

// Platform configurations based on current platform
const platforms = isMac ? [
  {
    name: 'mac_arm',
    platform: 'mac',
    architecture: 'arm64',
    revision: CHROME_VERSION
  },
  {
    name: 'mac_intel',
    platform: 'mac',
    architecture: 'x64',
    revision: CHROME_VERSION
  }
] : isWindows ? [
  {
    name: 'win_intel',
    platform: 'win32',
    architecture: 'x64',
    revision: CHROME_VERSION
  }
] : [];

const cacheDir = path.join(__dirname, '..', '.cache', 'puppeteer');

async function downloadChrome() {
  if (platforms.length === 0) {
    console.error('❌ Unsupported platform. This script only supports Mac and Windows.');
    process.exit(1);
  }

  console.log(`Starting direct Chrome download for current platform: ${currentPlatform.toUpperCase()}`);
  if (isMac) {
    console.log('Mac: Downloading both Intel (x64) and ARM (arm64) architectures');
  } else if (isWindows) {
    console.log('Windows: Downloading Intel (x64) architecture only');
  }
  
  // Create cache directory if it doesn't exist
  if (!fs.existsSync(cacheDir)) {
    fs.mkdirSync(cacheDir, { recursive: true });
  }

  const downloadUrls = {
    'mac_arm': `https://storage.googleapis.com/chrome-for-testing-public/138.0.7204.49/mac-arm64/chrome-mac-arm64.zip`,
    'mac_intel': `https://storage.googleapis.com/chrome-for-testing-public/138.0.7204.49/mac-x64/chrome-mac-x64.zip`,
    'win_intel': `https://storage.googleapis.com/chrome-for-testing-public/138.0.7204.49/win64/chrome-win64.zip`
  };

  // Define expected binary paths for each platform
  const binaryPaths = {
    'mac_arm': path.join(cacheDir, 'mac_arm', 'chrome-mac-arm64', 'Google Chrome for Testing.app', 'Contents', 'MacOS', 'Google Chrome for Testing'),
    'mac_intel': path.join(cacheDir, 'mac_intel', 'chrome-mac-x64', 'Google Chrome for Testing.app', 'Contents', 'MacOS', 'Google Chrome for Testing'),
    'win_intel': path.join(cacheDir, 'win_intel', 'chrome-win64', 'chrome.exe')
  };

  for (const platform of platforms) {
    const url = downloadUrls[platform.name];
    const binaryPath = binaryPaths[platform.name];
    
    // Check if binary already exists
    if (fs.existsSync(binaryPath)) {
      console.log(`✅ Chrome for ${platform.name} already exists at: ${binaryPath}`);
      console.log(`⏭️  Skipping download for ${platform.name}`);
      continue;
    }
    
    console.log(`\nDownloading Chrome for ${platform.name}...`);
    
    try {
      const platformDir = path.join(cacheDir, platform.name);
      if (!fs.existsSync(platformDir)) {
        fs.mkdirSync(platformDir, { recursive: true });
      }

      // Use curl to download (more reliable than Node.js http)
      const zipPath = path.join(platformDir, 'chrome.zip');
      const command = `curl -L "${url}" -o "${zipPath}"`;
      
      console.log(`Executing: ${command}`);
      execSync(command, { stdio: 'inherit' });
      
      // Extract the zip file
      console.log(`Extracting ${platform.name}...`);
      execSync(`unzip -q "${zipPath}" -d "${platformDir}"`, { stdio: 'inherit' });
      
      // Clean up zip file
      fs.unlinkSync(zipPath);
      
      console.log(`✅ Successfully downloaded and extracted Chrome for ${platform.name}`);
      
    } catch (error) {
      console.error(`❌ Failed to download Chrome for ${platform.name}:`, error.message);
      process.exit(1);
    }
  }
  
  console.log('\n🎉 Direct Chrome download process completed!');
}

downloadChrome(); 