var path = require("path");

const DesignOptionErrorCodes = {
  kDuplicateDesignOption: 75,
  kMissingDependencyDesignOptions: 77,
};
/**
 * iterate over each directory and get all the design option file path and save it to filePaths array.
 * @param directories - directories url
 * @param fsImpl - file system adapter
 * @param designOptionDirectoryApiInfo - design option api url and meta info
 * @returns -  return all the file path of design option files.
 */
/**
 * Interface for FSAdapter
 * @interface
 * Methods:
 * - readDir(): Array<string>
 * - isDirectory(): boolean
 * - readFile(): Object
 */

function getAllFilePathForDirectories(
  directories,
  fsImpl,
  designOptionDirectoryApiInfo
) {
  /**
   *
   * @param {*} respObj
   * @param {*} designOptionObj
   */
  function createOneEntry(respObj, name, filePath) {
    respObj.id = name;
    respObj.dirName = name;
    respObj.allowSelection = true;
    respObj.isToggleable = true;
    respObj.children = [];
    respObj.path = filePath;
  }

  const filePaths = [];
  const dirTree = {};

  directories.forEach((directory) => {
    if (designOptionDirectoryApiInfo[directory]?.isCustomDesignOption) {
      createOneEntry(dirTree, "designOptions", directory);
    }
  });
  /**
   * take parameter as path of directory and iterate over each directory inside it recursevly and
   *  save all the file path to the filePaths array.
   * @param dirPath - take the path of directory
   */
  function getAllFilePaths(
    dirPath,
    relativePath,
    isCustomDesignOption,
    dirTreeJson,
    parentTreeJson
  ) {
    try {
      const files = fsImpl.readDir(dirPath);
      let fileName, localeFileName, thumbnailFileName;
      files.forEach(function (file) {
        const filePath = path.join(dirPath, path.sep, file);
        const rPath = `${relativePath}/${file}`;
        const isDirectoryPath = fsImpl.isDirectory(filePath);
        if (isDirectoryPath) {
          let dirObj = {};
          if (isCustomDesignOption) {
            createOneEntry(dirObj, file, filePath);
            if (!dirTreeJson.children) {
              dirTreeJson.children = [];
            }
            dirTreeJson.children.push(dirObj);
          }
          getAllFilePaths(
            filePath,
            rPath,
            isCustomDesignOption,
            dirObj,
            dirTreeJson
          );
        } else {
          // TODO: revisit it as a part of Nomencalture
          switch (file) {
            case "designOption.json":
              if (isCustomDesignOption) {
                dirTreeJson.path = dirPath;
              }
            case "theme.json":
              fileName = file;
              break;
            case "localized.json":
              localeFileName = file;
              break;
            case "thumbnail.png":
              if (isCustomDesignOption) {
                thumbnailFileName = file;
              }
              break;
            /**
             * creating entry for each file object in treeview first then adding property to it later.
             * but when file name is designOption.json_backup, we need to ignore the treeview entry, but
             * treeview entry has already been created. So, we need to remove it.
             */
            case "designOption.json_backup":
              parentTreeJson.children.pop();
              break;
          }
        }
      });
      if (fileName)
        filePaths.push({
          dirPath,
          fileName,
          localeFileName,
          relativePath,
          isCustomDesignOption,
          dirObj: dirTreeJson,
          thumbnailFileName,
        });
    } catch (err) {}
  }
  directories.forEach((directory) =>
    getAllFilePaths(
      directory,
      designOptionDirectoryApiInfo[directory]?.apiUrl,
      designOptionDirectoryApiInfo[directory]?.isCustomDesignOption,
      dirTree,
      dirTree
    )
  );
  return { filePaths, dirTree };
}

/**
 * get all the design options json for a given design option paths and replace name with locale name and
 * return json as a promise.
 * @param filePaths - design option file paths
 * @param locale - locale to localized design option name.
 * @returns - return design options json as promise for all the file paths
 */
async function getDesignOptionJsonFromFilePath(filePaths, locale, fsImpl) {
  const uniqueIds = new Map();
  const designOptionThumbnailMap = {};
  const designOptions = await Promise.all(
    filePaths.map(
      ({
        fileName,
        localeFileName,
        dirPath,
        relativePath,
        isCustomDesignOption,
        dirObj,
        thumbnailFileName,
      }) => {
        const promiseArray = [];
        if (fileName) promiseArray.push(fsImpl.readFile(dirPath, fileName));
        if (localeFileName)
          promiseArray.push(fsImpl.readFile(dirPath, localeFileName));
        return Promise.all(promiseArray).then(async (fileBuffers) => {
          let jsonString = fileBuffers[0].toString();
          let thumbnailContent;
          if (fileBuffers[1]) {
            const localeObject = JSON.parse(fileBuffers[1]);
            if (localeObject[locale]) {
              for (let [key, value] of Object.entries(localeObject[locale])) {
                jsonString = jsonString.replace(`$$${key}`, value);
              }
            }
          }
          try {
            const designOptionJson = JSON.parse(jsonString);
            designOptionJson.dirPath = dirPath;
            designOptionJson.isCustomDesignOption = isCustomDesignOption;
            if (designOptionJson.thumbnail) {
              designOptionJson.thumbnailPath =
                relativePath + "/" + designOptionJson.thumbnail;
            }
            if (designOptionJson.isCustomDesignOption) {
              // thumbnail file name is coming from design option json, so we have to read file after parsing json.
              try {
                const thumbnailFileBuffer = await fsImpl.readFile(
                  dirPath,
                  designOptionJson.thumbnail
                );
                if (thumbnailFileBuffer) {
                  thumbnailContent = Buffer.from(thumbnailFileBuffer).toString(
                    "base64"
                  );
                }
              } catch (err) {
                thumbnailContent = null;
              }

              dirObj.designOptionId = designOptionJson.designOptionId;
              dirObj.designOptionName = designOptionJson.name;
              dirObj.type = designOptionJson.type;
              let depSet = new Set();
              if (
                designOptionJson.styles &&
                Array.isArray(designOptionJson.styles)
              ) {
                designOptionJson.styles.forEach((style) => {
                  if (style.designOptionId) {
                    depSet.add(style.designOptionId);
                  }
                });
              }
              dirObj.dependencies = Array.from(depSet);
              dirObj.errCodes = [];
              if (uniqueIds.has(dirObj.designOptionId)) {
                errCode = DesignOptionErrorCodes.kDuplicateDesignOption;
                dirObj.errCodes.push(errCode);
                existingEntry = uniqueIds.get(dirObj.designOptionId);
                if (
                  existingEntry.isCustomDesignOption &&
                  existingEntry.dirObj &&
                  !existingEntry.dirObj.errCodes.includes(errCode)
                ) {
                  existingEntry.dirObj.errCodes.push(errCode);
                }
              }
              designOptionThumbnailMap[
                designOptionJson.designOptionId
              ] = thumbnailContent;
              designOptionJson.dirObj = dirObj;
            }
            uniqueIds.set(designOptionJson.designOptionId, designOptionJson);
            return designOptionJson;
          } catch (err) {
            return null;
          }
        });
      }
    )
  );
  function getMissingDeps(designOption) {
    missingDeps = new Set();
    if (designOption.styles && Array.isArray(designOption.styles)) {
      designOption.styles.forEach((style) => {
        if (style.designOptionId && !uniqueIds.has(style.designOptionId)) {
          missingDeps.add(style.designOptionId);
        }
      });
    }
    return Array.from(missingDeps);
  }

  let newDesignOptions = designOptions.map((obj) => {
    newObj = { ...obj };
    if (newObj.isCustomDesignOption) {
      missingDeps = getMissingDeps(newObj);
      if (missingDeps.length > 0) {
        newObj.dirObj.errCodes.push(
          DesignOptionErrorCodes.kMissingDependencyDesignOptions
        );
        newObj.dirObj.missingDeps = missingDeps;
      }
    }
    return newObj;
  });

  return { newDesignOptions, designOptionThumbnailMap };
}

module.exports = {
  getAllFilePathForDirectories,
  getDesignOptionJsonFromFilePath,
};
