/*
 *  ADOBE CONFIDENTIAL
 *
 *  Copyright 2006-2016, Adobe Systems Incorporated
 *
 *  All Rights Reserved.
 *
 *  NOTICE:  All information contained herein is, and remains the property of
 *  Adobe Systems Incorporated and its suppliers, if any.  The intellectual and
 *  technical concepts contained herein are proprietary to Adobe Systems
 *  Incorporated and its suppliers and may be covered by U.S. and Foreign
 *  Patents, patents in process, and are protected by trade secret or copyright
 *  law.  Dissemination of this information or reproduction of this material is
 *  strictly forbidden unless prior written permission is obtained from Adobe
 *  Systems Incorporated.
 *
 *   Author: haridoss<ramadoss@adobe.com>, 24-July-2018
 */

let express = require("express");
let app = express();
let fs = require("fs");
let path = require("path");
let detectPort = require("detect-port");

let cpTempDirectoryPath = process.argv[2];
let qr = require("qr-image");
let os = require("os");

let cpPID = process.argv[3] || -1;
let webServerAddress, lastPreviewFolder, lifeTimeoutId;
let serverOutFilePath =
  cpTempDirectoryPath + "/" + "live_preview_server_address.txt";
let httpsServerOutFilePath = cpTempDirectoryPath + "/" + "https_server.cpinfo";

const MAX_SERVER_RETRY = 25;
let startAttemptCount = 0;

/*
    Default server lifetime is one hour.
    Configurable via ini file option - LivePreviewLifetime
    Minimum value is than 10 minutes - To avoid captivate reading it after server is started.
*/
let serverLifeTime =
  process.argv[4] && process.argv[4] > 600000 ? process.argv[4] : 3600000;

/*
    Default port is 25030.
    Configurable via ini file option - LivePreviewPort
    If value is less than 0, then it will be set to default value.
*/
let PORT = process.argv[5] && process.argv[5] > 0 ? process.argv[5] : 25030;

let netWorkMessage = process.argv[6] ? process.argv[6] : "";
let helpLinkText = process.argv[7] ? process.argv[7] : "Help";

let isHTTPS = false;

/************************************************************************************
 ********************************** ROUTE FUNCTIONS **********************************
 ************************************************************************************/

app.get("/livepreview/updateLastPreviewFolder", (req, res) => {
  lastPreviewFolder = req.query.folder;
  let svgFilePath = path.resolve(cpTempDirectoryPath + "/qr.svg");
  if (fs.existsSync(svgFilePath)) {
    fs.unlinkSync(svgFilePath);
  }
  res.send({ message: "success" });
});

app.get("/livepreview/isLivePreviewDirty", (req, res) => {
  if (req.query.folder != lastPreviewFolder) {
    res.send({
      isDirty: true,
      folder: lastPreviewFolder,
    });
  } else {
    res.send({
      isDirty: false,
    });
  }
});

app.get("/livepreview/isServerRunning", (req, res) => {
  res.send({ message: "success" });
});

app.get("/livepreview/getLivePreviewQRCode", (req, res) => {
  addLifetime();
  let respMsg = createQRSVG() ? "success" : "failure";
  res.send({ message: respMsg });
});

/************************************************************************************
 *********************************** SUB ROUTINES ************************************
 ************************************************************************************/

let run = () => {
  try {
    detectPort(PORT).then((iPort) => {
      if (iPort == PORT) {
        /* port is free. */
        console.log("looking for " + __dirname + "/server.key");
        console.log("looking for " + __dirname + "/server.crt");
        if (
          fs.existsSync(__dirname + "/server.key") &&
          fs.existsSync(__dirname + "/server.crt")
        ) {
          var https = require("https");
          var certOptions = {
            key: fs.readFileSync(__dirname + "/server.key"),
            cert: fs.readFileSync(__dirname + "/server.crt"),
          };
          https.createServer(certOptions, app).listen(PORT);
          postInit();
          isHTTPS = true;
          fs.writeFileSync(httpsServerOutFilePath, "");
        } else {
          app.listen(PORT, "0.0.0.0", function (error, data) {
            if (error) {
              console.log("Listen error: " + error.toString());
              return;
            }
            postInit();
            isHTTPS = false;
            try {
              if (fs.existsSync(httpsServerOutFilePath)) {
                fs.unlinkSync(httpsServerOutFilePath);
              }
            } catch (e) {
              console.log(e);
            }
          });
        }
      } else {
        /* try the next port until MAX_SERVER_RETRY */
        if (startAttemptCount < MAX_SERVER_RETRY) {
          ++startAttemptCount;
          ++PORT;
          run();
        } else {
          console.log("Failed to find port for live preview.");
        }
      }
    });
  } catch (err) {
    console.error(err);
    let logFilePath = cpTempDirectoryPath + "/" + "log.txt";
    fs.writeFileSync(logFilePath, err, (err) => {
      if (err) throw err;
    });
  }
};

let postInit = () => {
  writeWebserverAddress();
  bindWithCaptivate();
  addLifetime();
  app.use("/livepreview", express.static(path.resolve(cpTempDirectoryPath)));
};

let createQRSVG = () => {
  let svgFilePath = path.resolve(cpTempDirectoryPath + "/qr.svg");
  if (!fs.existsSync(svgFilePath)) {
    let urlToEncode =
      (isHTTPS ? "https://" : "http://") +
      webServerAddress +
      ":" +
      PORT +
      "/" +
      "livepreview/" +
      lastPreviewFolder +
      "/index.html" +
      "?previewFromQRCode=true";
    let svgObject = qr.svgObject(urlToEncode, { type: "svg" });
    let svgStr = frameSVG(
      svgObject.path,
      svgObject.size,
      urlToEncode,
      netWorkMessage
    );
    fs.writeFileSync(svgFilePath, svgStr);
  }
  return fs.existsSync(svgFilePath);
};

let writeWebserverAddress = () => {
  /*  Save the ipaddress to temp folder. 
        Reference : https://stackoverflow.com/questions/3653065/get-local-ip-address-in-node-js */

  let ifaces = os.networkInterfaces();
  Object.keys(ifaces).reverse().every(function (ifname) {
    return ifaces[ifname].every(function (iface) {
      if ("IPv4" !== iface.family || iface.internal !== false) {
        return true; /* iterate = true */
      }

      if (
        iface.address &&
        iface.address.indexOf("127.0.0.1") == -1 &&
        iface.address.indexOf("localhost") == -1
      ) {
        webServerAddress = iface.address;
        return false; /* iterate = false */
      }

      return true; /* iterate = true */
    });
  });

  if (fs.existsSync(serverOutFilePath)) {
    fs.unlinkSync(serverOutFilePath);
  }

  if (
    webServerAddress &&
    webServerAddress.indexOf("127.0.0.1") == -1 &&
    webServerAddress.indexOf("localhost") == -1
  ) {
    let serverOutStr = `${webServerAddress}\n${PORT}`;
    fs.writeFileSync(serverOutFilePath, serverOutStr, (err) => {
      if (err) {
        throw err;
      }
    });
  } else {
    /* Live preview works only with internet. */
    process.exit(1);
  }

  console.log("webServerAddress " + webServerAddress);
};

let bindWithCaptivate = () => {
  if (cpPID >= 0) {
    setInterval(function () {
      if (cpPID != process.ppid) {
        stopServer();
      }
    }, 60000 /* one minute. */);
  }
};

let addLifetime = () => {
  if (lifeTimeoutId) {
    clearTimeout(lifeTimeoutId);
  }
  lifeTimeoutId = setTimeout(stopServer, serverLifeTime);
};

let stopServer = () => {
  try {
    if (fs.existsSync(serverOutFilePath)) {
      fs.unlinkSync(serverOutFilePath);
    }
    if (fs.existsSync(httpsServerOutFilePath)) {
      fs.unlinkSync(httpsServerOutFilePath);
    }
  } catch (e) {
    console.log(e);
  }
  process.exit(0);
};

let frameSVG = (path, size, url, msg) =>
  `<svg xmlns='http://www.w3.org/2000/svg' xmlns:xlink="http://www.w3.org/1999/xlink" onload="customAlign()">
    <style type="text/css">
        a {
            cursor: pointer;
        }

        a text {
            fill: blue;
            text-decoration: underline;
        }

        a:hover, a:active {
        outline: dotted 2px blue;
        }

</style>

 <script type="text/javascript"><![CDATA[

   function customAlign(){
       var qr=document.getElementById("qr");
       var linktext=document.getElementById("linktext");
       var msgtext=document.getElementById("msgtext");
       var helptext=document.getElementById("helptext");

       var centerX = window.innerWidth/2;
       var centerY = window.innerHeight/2;
       var width = 300;
       var height = 300;
       var x = centerX-width/2;
       var y = (centerY-height/2) - 150;

       qr.setAttribute("x", x);
       qr.setAttribute("y", y);

       var bounds = linktext.getBoundingClientRect();
       width = bounds.width;
       
       x = centerX-width/2;
       y = y + height + 35;
       linktext.setAttribute("x", x);
       linktext.setAttribute("y", y);

       y = y + 35;
       msgtext.setAttribute("x", x);
       msgtext.setAttribute("y", y);
       
       y = y + 35;
       
       helptext.setAttribute("x", x);
       helptext.setAttribute("y", y);
   }
   window.addEventListener("resize", customAlign);
  ]]></script>
    
    <svg id="qr"  viewBox='0 0 ${size} ${size}' width='117px' height='117px' >
        <path d='${path}' />
    </svg>
</svg>
`;

/************************************************************************************
 ********************************** MAIN *********************************************
 ************************************************************************************/
run();
