/*
 *  ADOBE CONFIDENTIAL
 *
 *  Copyright 2006-2024, Adobe Systems Incorporated
 *
 *  All Rights Reserved.
 *
 *  NOTICE:  All information contained herein is, and remains the property of
 *  Adobe Systems Incorporated and its suppliers, if any.  The intellectual and
 *  technical concepts contained herein are proprietary to Adobe Systems
 *  Incorporated and its suppliers and may be covered by U.S. and Foreign
 *  Patents, patents in process, and are protected by trade secret or copyright
 *  law.  Dissemination of this information or reproduction of this material is
 *  strictly forbidden unless prior written permission is obtained from Adobe
 *  Systems Incorporated.
 *
 *   Author: Nitesh Bhargava<nbhargava@adobe.com>, 24-Nov-2023
 */
require('../../polyfills.js');
const { AdobeAsset, DirectoryMediaType, Directory, Properties, compositeXfer, getCompositeComponentPresignedUrl,
	LinkRelation, streamToGetSliceCallback, parseLinksFromResponseHeader } = require('@dcx/assets');
const mime = require('mime');
var path = require("path");
const fs = require("fs")
const NodeCache = require('node-cache');
const { generateMD5Hash, createHTTPServiceSetAPIKey, getPlatFormUrl, getCCreviewURL } = require("./utils");
const {
	createRepoAPISession,
	createComposite,
	AdobeRepoAPISession,
	pullCompositeManifestOnly,
	pushComposite,
	uploadNewComponent,
	AdobeDCXComposite,
	uploadComponent,
	pullCompositeVersionManifestOnly,
	newDCXComposite,
	copyAssetForNewComponent,
	newCompositeAsCopyOf,
	AdobeDCXBranch,
	AdobeDCXError,
	AdobeHTTPService
} = require('@dcx/dcx-js');

const CC_REVIEW_SERVICE_URL = getCCreviewURL("prod");
const PLATFORM_URL = getPlatFormUrl("prod");
const cache = new NodeCache();
function logErrorResponse(error) {
	if (isAdobeDCXError(error) && error.response) {
		const { statusCode, message, response, headers: { ['x-request-id']: xRequestId }, xhr } = error.response ?? {};
		if (xhr) {
			console.error('Request Body: ', Buffer.from(xhr.body).toString('utf-8'));
		}
		console.error(statusCode, message, response?.message, xRequestId);
	} else {
		console.error(error?.message ?? "Unexpected Error");
	}
}

function readFileToBuffer(filePath) {
	return new Promise((resolve, reject) => {
		fs.readFile(filePath, (err, data) => {
			if (err) {
				reject(err);
			} else {
				resolve(data);
			}
		});
	});
}


function isAdobeDCXError(p) {
	if (!p || typeof p !== 'object') {
		return false;
	}

	return p.name === 'AdobeDCXError';
}

const createCloudComposite = async (session, parentDir, name) => {
	//Composite #(https://developers.corp.adobe.com/supplemental-resources/docs/glossary.md#composite)
	//a.create a cloud composite place holder 
	const composite = await createComposite(session, newDCXComposite(undefined, parentDir.repositoryId, name, undefined, 'application/vnd.adobe.captivate.sharedartifact+dcx'), parentDir, name);
	//b.push once to upload the manifest at first time
	await pushComposite(session, composite, true);
	return composite;
}

const readCloudComposite = async (session, { repositoryId = "", assetId = "" }) => {
	const composite = newDCXComposite(assetId, repositoryId);
	const branch = await pullCompositeManifestOnly(session, composite);
	composite.resolvePullWithBranch(branch);
	await Promise.all(
		branch.allComponents().map((component) =>
			session.getCompositeComponent(component, component.id, component.version, 'arraybuffer'))
	);
}

const uploadNewComponentForComposite = async (session, composite, name, path, data, type) => {
	//console.log("path: " + path);
	const md5_string = await generateMD5Hash(data);
	console.log("md5_string " + name + " : " + md5_string);
	const uploadResult = await uploadNewComponent(session, composite, data, type, undefined, undefined, md5_string);
	composite.current.addComponentWithUploadResults(name, undefined, path, composite.current.rootNode, uploadResult);
	const [componentId] = Object.keys(uploadResult.records);
	return componentId;
}

const updateComponentFromAFilePath = async (session, composite, filePath, commonString, manifest, service, id) => {
	let mime_type = mime.getType(filePath);
	let buffer = await readFileToBuffer(filePath);
	let name = path.basename(filePath);
	const commonLength = commonString.length;
	//console.log("update filepath " + filePath + " mimetype  "+ mime_type)
	if (filePath.indexOf('.DS_Store') >= 0) return;
	if (mime_type == null) {
		mime_type = 'application/zip';
	}
	let component_id = "0";
	let isAlreadyPresent = false
	const relativePath = filePath.substring(commonLength);
	manifest.forEach(async element => {
		const componentData = JSON.parse(JSON.stringify(element))
		if (componentData.path == relativePath) {
			isAlreadyPresent = true;
			const md5 = await generateMD5Hash(buffer);
			component_id = componentData.id
			if (md5 != componentData.md5) {
				console.log(md5 + " md5 " + componentData.md5)
				console.log(component_id);
				const component = await composite.current.getComponentWithId(component_id)
				console.info(`before update component, the version is ${component.version}`);
				//component.assetId = component_id
				const uploadResult = await uploadComponent(session, composite, component, buffer)
				composite.current.updateComponentWithUploadResults(componentData, uploadResult);
				await pushComposite(session, composite, true);
				composite.acceptPush();
				//read the metadata of component from local manifest, assert the version got incremented. 
				composite.current.getComponentWithId(component_id);
			}
			//console.log("buffer length " + buffer.byteLength + " fd  " + componentData.length)

		}
	});
	if (!isAlreadyPresent) {
		console.log("new file : " + filePath)
		await uploadComponentFromAFilePath(session, composite, filePath, commonString);
	}

}
/**
 * Recursively search across each page for an asset with a given name.
 * @param predicate 
 * @param paged 
 * @returns 
 */
async function findInPagedByName(paged, nameToFind) {
	const compare = typeof nameToFind === 'string' ? (asset) => asset.name === nameToFind : (asset) => nameToFind.test(asset.name ?? '');
	return await findInPaged(paged, compare);
}

/**
 * Recursively search across each page applying a predicate function and returning the result if found.
 * @param predicate 
 * @param paged 
 * @returns 
 */
async function findInPaged(paged, predicate) {
	while (paged.items.length) {
		const found = paged.items.find(predicate);
		if (found) {
			return found;
		}
		if (!paged.hasNextPage()) {
			return;
		}
		await paged.getNextPage();
	}
}

const getAssignedDirectoryByName = async (session, dirname) => {
	// Reference the appropriate storage model (USM/ESM) here:
	// https://git.corp.adobe.com/pages/caf/api-spec/chapters/storage_repo/storage_repository.html
	// Assinged Directories# (https://developers.corp.adobe.com/index-repository-api/docs/index-document.md#assigned-directories)
	const { result: { assignedDirectories } } = await session.getIndexDocument().catch((err) => {
		throw new Error(`[dcx-js-demo] Bad response on discovering index doc because ${err}`);
	});

	if (!Array.isArray(assignedDirectories) || assignedDirectories.length === 0) {
		throw new Error('[dcx-js-demo] there are no directories assigned to this user');
	}

	const requestedDir = await session.getDirectory(assignedDirectories[0], { type: DirectoryMediaType })
		.catch((err) => {
			throw new Error(`[dcx-js-demo] Could not load Index Repository ${err}`);
		}).then(({ paged }) => findInPagedByName(paged, dirname))

	if (!requestedDir) {
		throw new Error('[dcx-js-demo] Could not locate the requested directory');
	}

	return requestedDir;
}

const createDirectory = async (session, svc, parentDir, name) => {
	try {
		const { result } = await session.createAsset(parentDir, name, false, DirectoryMediaType);
		return new Directory(result, svc);
	} catch (err) {
		// check for an existing directory
		const parent = parentDir instanceof Directory ? parentDir : new Directory(parentDir, svc);
		const { result: { children: siblings } } = await parent.getPagedChildren();
		const existingDir = siblings.find((asset) => asset[Properties.REPO_NAME] === name);
		if (!existingDir) throw err; // no existing directory found -- rethrow
		return new Directory(existingDir, svc);
	}
}

const uploadComponentFromAFilePath = async (session, composite, filePath, commonString) => {
	let mime_type = mime.getType(filePath);
	let buffer = await readFileToBuffer(filePath);
	let name = path.basename(filePath);
	const commonLength = commonString.length;
	//console.log("filepath " + filePath + " mimetype  "+ mime_type)
	if (filePath.indexOf('.DS_Store') >= 0) return;
	if (mime_type == null) {
		mime_type = 'application/zip';
	}
	const componentId = await uploadNewComponentForComposite(session, composite, name, filePath.substring(commonLength), buffer, mime_type);
	return componentId;
}

const getFiles = async (dir, files) => {
	// Get an array of all files and directories in the passed directory using fs.readdirSync
	const fileList = fs.readdirSync(dir);
	// Create the full path of the file/directory by concatenating the passed directory and file/directory name
	for (const file of fileList) {
		let name = `${dir}/${file}`;
		// Check if the current file/directory is a directory using fs.statSync
		if (fs.statSync(name).isDirectory()) {
			// If it is a directory, recursively call the getFiles function with the directory path and the files array
			getFiles(name, files);
		} else {
			// If it is a file, push the full path to the files array
			files.push(name);
		}
	}
	return files;
}

const uploadReview = async (authToken, name, path, appKey, projectID, projectName) => {
	if(!path){
		const error = new Error("Publish folder path is missing");
		throw error;
	}
	try {
		const service = createHTTPServiceSetAPIKey(appKey, authToken);
		const session = createRepoAPISession(service, PLATFORM_URL);
		var projectDirAssetId = await findProjectDirectory(authToken, appKey, projectID);
		const indexRepoId = await fetchIndexRepoId(session)
		if (!projectDirAssetId) {
			console.log("project directory not yet provisioned")
			projectDirAssetId = await provisionReviewDirectory(authToken, appKey, projectID)
			directoryAssetIdCachedMap[projectID] = projectDirAssetId
		}
		const dirAsset = {
			repositoryId: indexRepoId,
			assetId: projectDirAssetId
		};
		var projectDir = new Directory(dirAsset, service)


		console.info("create cloud composite");
		let composite = await createCloudComposite(session, projectDir, name);

		console.info("read composite");
		await readCloudComposite(session, composite);
		await pushComposite(session, composite, true);
		composite.acceptPush();
		return { assetId: composite.assetId, projectDirectoryAssetId: projectDir.assetId };
	}
	catch (error) {
		console.log(error);
		throw error;
	}

}

const uploadComponents = async (authToken, id, path, appKey, projectID, projectName) => {
	const service = createHTTPServiceSetAPIKey(appKey, authToken);
	const session = createRepoAPISession(service, PLATFORM_URL);
	var projectDirAssetId = await findProjectDirectory(authToken, appKey, projectID);

	try {
		const indexRepoId = await fetchIndexRepoId(session)
		const composite = newDCXComposite(id, indexRepoId)

		let assetPath = path
		filesInDir = await getFiles(assetPath, new Array());
		//console.log(filesInDir);
		startTime = Date.now()
		const pulledComposite = await pullCompositeManifestOnly(
			session,
			composite
		);
		composite.resolvePullWithBranch(pulledComposite);
		const manifest = await session.getCompositeManifest(composite)

		let noOfFiles = filesInDir.length;
		await Promise.all(filesInDir.map(async (filePath) => {
			await uploadComponentFromAFilePath(session, composite, filePath, assetPath + "/");
			let count = cache.get(`${id}-count`);
			if (count === undefined) {
				count = 0;
			}
			count = count + 1;
			let progress = parseInt(count / noOfFiles * 100)
			cache.set(`${id}-count`, count);
			cache.set(`${id}-progress`, progress)
		}
		));

		// thumbnail logic
		/*let buffer = await readFileToBuffer("/Users/niteshbhargava/workspace/CaptivateNext/dev/extras/xplat/mul/[INSTALLDIR]/ns/patterns.png");        
		const uploadResult = await uploadNewComponent(session, composite, buffer, "image/png");
		composite.current.addComponentWithUploadResults("thumbnail", 'thumbnail', "thumbanail.png", composite.current.rootNode, uploadResult);        
		const uploadResult1 = await uploadNewComponent(session, composite, buffer, "image/png");
		composite.current.addComponentWithUploadResults("preview", 'preview', "preview.png", composite.current.rootNode, uploadResult1);*/


		await pushComposite(session, composite, true);
		composite.acceptPush();
		const { result: repoMetadata } = await session.getRepoMetadata(composite)
		return { assetId: composite.assetId, lastUpdatedOn: repoMetadata.modifyDate};
	} catch (error) {
		console.log(error);
		throw error;
	}
}

const updateReview = async (authToken, id, path, appKey) => {


	const service = createHTTPServiceSetAPIKey(appKey, authToken);
	const session = createRepoAPISession(service, PLATFORM_URL);

	try {
		const indexRepoId = await fetchIndexRepoId(session)
		const composite = newDCXComposite(id, indexRepoId)

		assetPath = path
		filesInDir = await getFiles(assetPath, new Array());
		//console.log(filesInDir);
		startTime = Date.now()
		const pulledComposite = await pullCompositeManifestOnly(
			session,
			composite
		);
		composite.resolvePullWithBranch(pulledComposite);
		const manifest = await session.getCompositeManifest(composite)
		const manifestComponentData = manifest.manifestData.components
		await Promise.all(filesInDir.map(async (filePath) => {
			await updateComponentFromAFilePath(session, composite, filePath, assetPath + "/", manifestComponentData, service, id);

		}
		));
		endTime = Date.now();
		await pushComposite(session, composite, true);
		//composite.acceptPush();
		endTime = Date.now();
		console.log("Time taken to update " + assetPath + " : ", endTime - startTime)

		const { result: repoMetadata } = await session.getRepoMetadata(composite)
		return { assetId: composite.assetId, lastUpdatedOn: new Date()};
		
	} catch (error) {
		console.log(error);
		throw error;
	}
}

async function fetchCollab(url, token, appKey) {
	//console.log(url);
	try {
		const response = await fetch(url, {
			method: "GET",
			headers: {
				Authorization: `Bearer ${token}`,
				"x-api-key": appKey,
			},
		});

		if (!response.ok) {
			throw new Error(`Could not fetch collaborators. Status: ${response.status}`);
		}

		const data = await response.json();
		return data;
	} catch (error) {
		console.error("Error fetching data:", error.message);
		throw error;
	}
}

const getCollaboratorListForUrns = async (urns, token, appKey) => {
	//let collabs = []
	await Promise.all(urns.map(async function (urn) {
		const assetURN = urn["id"];
		const response = await fetchCollab(
			`https://invitations.adobe.io/api/v4/share/${assetURN}`,
			token,
			appKey
		);
		const collabsArray = response["collaborators"];
		let collabs = []
		collabsArray.forEach((collaborator) => {
			let collab = {}
			if (collaborator.role != "owner") {

				collab["CollaboratorId"] = collaborator.id;
				collab["additionalData"] = collaborator.additionalData;
			}
			collabs.push(collab);
		});
		const inviteesArray = response["invitations"];
		inviteesArray.forEach((invitee) => {
			let collab = {}
			collab["CollaboratorId"] = invitee.email;
			collab["additionalData"] = invitee.additionalData;
			collabs.push(collab);
		});
		urn["collaborators"] = collabs;
	}));
	return urns;
};

const listReviews = async (service, session, token, assetId, appKey) => {
	const indexRepoId = await fetchIndexRepoId(session)

	const dirAsset = {
		repositoryId: indexRepoId,
		assetId: assetId
	};

	const projectDirectory = new Directory(dirAsset, service);
	const children = await projectDirectory.getPagedChildren();
	//console.log(children.result.children);
	const urns = [];
	children.result.children.forEach((element) => {
		urns.push({
			id: element["repo:id"],
			name: element["repo:name"],
			lastUpdatedOn: element["repo:modifyDate"],
		});
	});
	//console.log(urns);
	const collabs = await getCollaboratorListForUrns(urns, token, appKey);

	return collabs
	//console.log("Time taken to update " + " : ", endTime - startTime)
};

let directoryAssetIdCachedMap = {};  //map of project GUID vs project directory asset ID
let indexRepoIdCached = null;

const fetchIndexRepoId = async (session) => {
	if(indexRepoIdCached) return indexRepoIdCached;
	const { result: { assignedDirectories } } = await session.getIndexDocument();
	const index = assignedDirectories && assignedDirectories[0];
	indexRepoIdCached = index.repositoryId;
	return index.repositoryId;
}

const findProjectDirectory = async (token, appKey, projectUUID) => {
	if (directoryAssetIdCachedMap[projectUUID]) return directoryAssetIdCachedMap[projectUUID];
	var directoryAssetId = null;
	const url = `${CC_REVIEW_SERVICE_URL}/outputs?documentId=${projectUUID}`
	console.log("url: " + url)
	var data;
	try {
		const response = await fetch(url, {
			method: "GET",
			headers: {
				Authorization: `Bearer ${token}`,
				"x-api-key": appKey,
			},
		});

		if (!response.ok) {
			throw new Error(`Error while finding project directory. Status: ${response.status}`);
		}
		if (response.status == 204) {
			directoryAssetIdCachedMap[projectUUID] = directoryAssetId;
			return directoryAssetId;
		}
		data = await response.json();
		//return data;
	} catch (error) {
		console.log("Error fetching data:", error.message);
		throw error;
	}
	directoryAssetId = data["repo:assetId"];
	directoryAssetIdCachedMap[projectUUID] = directoryAssetId;
	return directoryAssetId;
};

const provisionReviewDirectory = async (token, appKey, projectID) => {
	const url = `${CC_REVIEW_SERVICE_URL}/output/:provision?documentId=${projectID}`
	try {
		const response = await fetch(url, {
			method: "POST",
			headers: {
				Authorization: `Bearer ${token}`,
				"x-api-key": appKey,
			},
		});

		if (!response.ok) {
			throw new Error(`Error while provisioning review directory. Status: ${response.status}`);
		}

		const data = await response.json();
		return data["repo:assetId"];
	} catch (error) {
		console.log("Error fetching data:", error.message);
		throw error;
	}
}

async function renameAsset(session, assetId, newName) {
	const indexRepoId = await fetchIndexRepoId(session)
	const asset = newDCXComposite(assetId, indexRepoId)
	const { result: repoMetadata } = await session.getRepoMetadata(asset)
	const oldName = repoMetadata.path?.split('/').pop();
	const newPath = repoMetadata.path?.replace(oldName, newName);
	const result = await session.moveAsset(asset, { repositoryId: indexRepoId, path: newPath }, false, false);
	return result;
}

function getCacheKey(key) {
	let data = 0;
	if (cache.has(key)) {
		data = cache.get(key);
	} else {
		return 0;
	}
	return data;
}

async function discardAsset(session, token, apiKey, assetId) {
	const indexRepoId = await fetchIndexRepoId(session)
	const asset = {assetId: assetId, repositoryId: indexRepoId}
	const url = await session.getLinkHrefForAsset(asset, LinkRelation.DISCARD);
	const response = await fetch(url, {
		method: "POST",
		headers: {
			Authorization: `Bearer ${token}`,
			"x-api-key": apiKey,
		},
	});

	if (!response.ok) {
		throw new Error(`Error while discarding asset. Status: ${response.status}`);
	}
	console.log(response) 
	return "success";
}
module.exports = { uploadReview, updateReview, listReviews, findProjectDirectory, renameAsset, getCacheKey, discardAsset, uploadComponents };

