/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2025 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 *************************************************************************
 */

const fs = require("fs");
const path = require("path");
const os = require("os");
const fg = require("fast-glob");
const fontkit = require("fontkit");

/**
 * Get a list of font directories to scan based on the platform.
 * 
 * @returns {string[]} List of absolute paths to scan
 */
function getSearchPaths() {
  const platform = process.platform;
  const home = os.homedir();

  if (platform === "darwin") {
    return [
      "/System/Library/Fonts",
      "/Library/Fonts",
      `${home}/Library/Fonts`,
      "/Network/Library/Fonts",
      `${home}/Library/Application Support/Adobe/CoreSync/plugins/livetype/.r`,
      `${home}/Library/Application Support/Adobe/CoreSync/plugins/livetype/r`,
      `${home}/Library/Application Support/Adobe/Fonts`,
      `${home}/Library/Caches/Adobe/TypeSupport/Database`,
      `${home}/Library/Caches/com.adobe.acc.AdobeCreativeCloud/fonts`,
      "/Applications/Microsoft Word.app/Contents/Resources/Fonts",
      "/Applications/Microsoft PowerPoint.app/Contents/Resources/Fonts",
      `${home}/Library/FontExplorer X/Font Library`,
      `${home}/Library/Extensis/Suitcase Fusion/Font Vault`,
      `${home}/Library/Application Support/Google/Chrome/Default/Extensions/*/fonts`,
    ];
  } else if (platform === "win32") {
    return [
      path.join(process.env.WINDIR || "C:\\Windows", "Fonts"),
      path.join(home, "AppData/Local/Microsoft/Windows/Fonts"),
      path.join(home, "AppData/Roaming/Adobe/CoreSync/plugins/livetype/.r"),
      path.join(home, "AppData/Roaming/Adobe/CoreSync/plugins/livetype/r"),
      path.join(home, "AppData/Roaming/Adobe/Fonts"),
      "C:/Program Files/Common Files/Adobe/Fonts",
      "C:/Program Files (x86)/Common Files/Adobe/Fonts",
    ];
  } else {
    return [
      "/usr/share/fonts",
      "/usr/local/share/fonts",
      `${home}/.fonts`,
      `${home}/.local/share/fonts`,
      `${home}/.var/app/*/data/fonts`,
      "/var/lib/flatpak/app/*/current/active/files/share/fonts",
      "/opt/*/fonts",
      "/usr/share/fonts-*",
    ];
  }
}

/**
 * Recursively find all files inside a directory using fast-glob.
 * 
 * @param {string} dir - Directory to search
 * @returns {Promise<string[]>} Paths of found files (not yet validated as fonts)
 */
async function findFontsInDirectory(dir) {
  const pattern = path.join(dir, `**/*`).replace(/\\/g, "/");
  try {
    const files = await fg(pattern, {
      onlyFiles: true,
      suppressErrors: true,
      unique: true,
      dot: true,
      followSymbolicLinks: true,
    });

    const validFiles = files.filter(f => {
      try {
        return fs.existsSync(f) && fs.statSync(f).isFile();
      } catch {
        return false;
      }
    });
    return validFiles;
  } catch (err) {
    return [];
  }
}

/**
 * Aggregate all font file paths from all discovered font directories.
 * 
 * @returns {Promise<string[]>} Unique list of font file paths
 */
async function getSystemFontPaths() {
  const paths = new Set();
  const dirs = getSearchPaths();

  for (const dir of dirs) {
    try {
      fs.accessSync(dir, fs.constants.R_OK);
      const fonts = await findFontsInDirectory(dir);
      fonts.forEach(f => paths.add(f));
    } catch (e) {
    }
  }

  return Array.from(paths);
}

/**
 * Categorize font type based on family name.
 * 
 * @param {string} family - Font family name
 * @returns {string} One of: monospace, serif, sansSerif, cursive, fantasy, unknown
 */
function classifyType(family) {
  const name = family.toLowerCase();
  if (name.includes("mono") || name.includes("code") || name.includes("courier") || name.includes("console") || name.includes("typewriter")) return "monospace";
  if (name.includes("serif") && !name.includes("sans")) return "serif";
  if (name.includes("sans")) return "sansSerif";
  if (name.includes("script") || name.includes("cursive")) return "cursive";
  if (name.includes("fantasy") || name.includes("decorative")) return "fantasy";
  return "unknown";
}

/**
 * Safely sanitize and clamp font weight between 1 and 1000.
 * Returns 400 if the input is invalid.
 *
 * @param {any} weight - The raw weight value from the font metadata.
 * @returns {number} A valid CSS font weight.
 */
function sanitizeFontWeight(weight) {
  const n = Number(weight);
  return Number.isFinite(n) && n >= 1 && n <= 1000 ? Math.round(n) : 400;
}


/**
 * Extract metadata from a given font file using fontkit.
 * 
 * @param {string} fontPath - Path to font file
 * @returns {Promise<Object[] | null>} Parsed font entries or null on failure
 */
async function extractFontInfo(fontPath) {
  try {
    const font = await fontkit.open(fontPath);
    const families = font.fonts || [font];
    const entries = [];

    for (const f of families) {
      if (!f.familyName || !f.subfamilyName) {
        continue;
      }
      entries.push({
        family: f.familyName,
        path: fontPath,
        style: f.subfamilyName.toLowerCase().replace(/\s+/g, ""),
        weight: sanitizeFontWeight(f.weight),
        type: classifyType(f.familyName),
      });
    }

    return entries.length > 0 ? entries : null;
  } catch (e) {
    return null;
  }
}

/**
 * Main entry to scan and categorize system fonts.
 * 
 * @async
 * @function
 * @returns {Promise<Object>} Font families with associated font files and metadata
 */
async function getSystemFonts() {
  const paths = await getSystemFontPaths();
  const mapping = {};
  let count = 0;

  for (const p of paths) {
    const entries = await extractFontInfo(p);
    if (!entries) continue;

    for (const e of entries) {
      mapping[e.family] = mapping[e.family] || [];
      mapping[e.family].push({
        path: e.path,
        style: e.style,
        type: e.type,
        weight: e.weight,
      });
      count++;
    }
  }
  return mapping;
}

module.exports = { getSystemFonts };