/********************************************************************/
/*                                                                  */
/* ADOBE SYSTEMS INCORPORATED                                       */
/* Copyright 1986 - 2005 Adobe Systems Incorporated                 */
/* All Rights Reserved                                              */
/*                                                                  */
/* NOTICE:  Adobe permits you to use, modify, and distribute this   */
/* file in accordance with the terms of the Adobe license agreement */
/* accompanying it.  If you have received this file from a source   */
/* other than Adobe, then your use, modification, or distribution   */
/* of it requires the prior written permission of Adobe.            */
/*                                                                  */
/********************************************************************/
/* 
 *	Program Name:                                                       
 *		wordcnt                                                         
 *                                                                     
 *	General Description:                                                
 *		Wordcnt counts words in the current document.                  
 *		It operates by locating all flows on body pages in the           
 *		document, and then reading the content of every paragraph        
 *		in each flow.  It then counts the words in each paragraph        
 *		A characters is defined to be either an "in-word" or a           
 *		"not-in-word" character.  A "word" is a group of one or          
 *		more "in-word" characters - delimited on either side of the      
 *		group by a "not-in-word" character, a paragraph break, or        
 *		a hard (user-inserted) line break.                               
 *		In addition the program counts words in graphic text lines       
 *		which appear on body pages of the document.                      
 *		Text in footnotes, tables, or generated from variables or        
 *		cross references is counted.                                     
 *		Text generated by autonumber strings or from master or           
 *		reference pages (e.g. headers and footers) is not counted.       
 *                                                                     
 *	Invocation:                                                         
 *		Wordcnt is invoked as a document report.  From the             
 *		Document Reports dialog, choose "wordcnt"                      
 *                                                                     
 *	Install Info (UNIX):                                                       
 *		If you compile this as an RPC client (the default),
 *		add these lines to the apiclients file:
 *
 *		<APIclient                                                       
 *			<Name wordcnt>                                              
 *			<Type Report>                                                 
 *			<Description Count the number of words in the document >      
 *			<Directory fdk_install_dir/samples/wordcnt>
 *			<CommandLine $FAPIDIR/fa.wordcnt>                       
 *		>                                                                
 *
 *		Replace fdk_install_dir with the path of the directory in
 *		which you installed your copy of the FDK files.
 *
 *		If you compile this as a dynamic client, change the line
 *			<CommandLine $FAPIDIR/wordcnt>
 *		to:
 *			<SharedLibrary fa.wordcnt.dl>
 *
 *		Restart maker.
 *
 *	Install Info (Windows):
 *		Add the following entry (all on one line) to the [APIClients]
 *		section of your maker.ini file:
 *
 *		wordcnt=DocReport,Word Count Document Report,
 *			fdk_install_dir\samples\wordcnt\debug\wordcnt.dll, all
 *
 *		Replace fdk_install_dir with the path of the directory
 *		in which you installed your copy of the FDK files.
 *		Restart maker. 
 *
 *	Exceptions:                                                         
 *		Wordcount count words in documents only - it cannot be           
 *		invoked to count the number of words in a book.                  
 *                                                                     
 *********************************************************************/      

#include "wordcnt.h"
#include "fstrings.h"
#include "fdetypes.h"

static UCharT copyright[] = "Copyright (C) 1986-1995 Frame Technology Corporation. All rights reserved.";

/*
* F_ApiInitialize()
* Called from api each time wordcount is invoked by user from document
* report.  This is effectively the "main" function for the client.
* It calls CountWordsInDoc() to count the number of words in the 
* document and reports the result to the user. Then it bails out.
*/
VoidT F_ApiInitialize(init)
IntT init;
{
	F_ObjHandleT docId;
	IntT count, langnum; 
	UByteT str[200];

	/*Unicode enabled. */
	F_FdeInit();
	F_ApiEnableUnicode(True);
  	F_FdeInitFontEncs("UTF-8");

	if (init != FA_Init_DocReport)
		goto done;

	docId = F_ApiGetId(0, FV_SessionId, FP_ActiveDoc);
	if (!docId)
	{
		F_ApiAlert("No current doc", 0);
		goto done;
	}

	count = CountWordsInDoc(docId);

	/* 
	 * Output the final word count 
	 * This is one way to localize clients for multiple language versions
	 * of maker/builder.  This is really only useful for clients with a 
	 * very small number of strings.
	 */
	str[0] = '\0';
	langnum = F_ApiGetInt(0, FV_SessionId, FP_Language);
	switch (langnum)
	{
	case FV_LANG_GERMAN:      /* 232 = o dieresis */
	case FV_LANG_NEW_GERMAN:
	case FV_LANG_SWISS_GERMAN:
	case FV_LANG_NEW_SWISS_GERMAN:
		SprintfInt (str, "%d W\232rter im Dokument.", count);
		break;
	case FV_LANG_FRENCH:
	case FV_LANG_CANADIAN_FRENCH:
		SprintfInt (str, "Nombre de mots dans le document: %d", count);
		break;
	case FV_LANG_ITALIAN:   /* 217 = e grave */
		SprintfInt (str, "Il numero di parole nel documento \217 %d.", count);
		break;
	case FV_LANG_SWEDISH:
		SprintfInt (str, "%d ord i dokumentet.", count);
		break;
	case FV_LANG_BRITISH:
	case FV_LANG_ENGLISH:
		SprintfInt (str, "%d words in document.", count);
		break;
	case FV_LANG_SPANISH:
		SprintfInt (str, "%d palabras en el documento.", count);
		break;
	case FV_LANG_BRAZILIAN:
		SprintfInt (str, "%d palavras no documento.", count);
		break;
	default: /* shouldn't happen - just in case other language used */
		SprintfInt (str, "%d", count);
		break;
	}

	F_ApiAlert(str,FF_ALERT_CONTINUE_NOTE);

done:
	/* we're done - keel over and die! */
	F_ApiBailOut(); 
}

