
/**
 * Class backing the eventlist component 
 * @param componentName current component node name (currentNode.getName())
 * @param componentPath path to the component used to form URL to target this component's server side scripts 
 * @param tagsMap Map of all tags in form {tagId:tagName}
 * @param filterTagArray List of filter tags configured in Filter Configuration dialog
 * @return
 */

function Eventlist (componentName, componentPath, tagsMap, filterTagArray, folderId, viewTypeCookieName, campaignId){
    
    this.ViewType = {
        list : "list",
        calendar : "calendar"
    }
    
    this.viewTypeCookieName = viewTypeCookieName;
    
    this.tagsMap = tagsMap;
    this.filterTagArray = filterTagArray;
    
    this.currentSearchTag = undefined;  
    
    this.pageParams = "?isRepositoryAjaxCall=true";
    if ( folderId != null ) {
        this.pageParams += "&folder-id=" + folderId;
    }
    if( campaignId != 'null' ) {
        this.pageParams = this.pageParams + "&campaign-id=" + campaignId;
    }
    
    
    var defViewType = getCookie(this.viewTypeCookieName);
    this.currentView = ( defViewType == this.ViewType.calendar ? this.ViewType.calendar : this.ViewType.list );
    this.currentPage = 0;

    //state for calendar view
    //initialize current date
    this.currentDate = new Date();

    //cache HTML elements
    var eventListContainer  = CQ.Ext.get("eventlist_" + componentName);
    var listPanel =  CQ.Ext.get('listPanel_' + componentName);
    var listViewEl = eventListContainer.select(".eventlist-listview a").first();
    var dateViewEl = eventListContainer.select(".eventlist-dateview a").first();
    
    this.getComponentName = function() {
        return componentName;
    }
    
    this.getComponentPath = function() {
        return componentPath;
    }
    
    this.getEventListContainer = function() {
        return eventListContainer;
    }
    
    this.getListPanel = function() {
        return listPanel;
    }
    
    this.getListViewElement = function(){
        return listViewEl;
    }
    
    this.getDateViewElement = function(){
        return dateViewEl;
    }
    
}

Eventlist.prototype = {
    
    getPageSize : function(){
        var elem  = CQ.Ext.get("selectPageSize_" + this.getComponentName());
        return elem.getValue();
    },
    
    getCurrentYear : function() {
        return this.currentDate.getFullYear();
    }, 
    
    getCurrentMonth : function() {
        return this.currentDate.getMonth();
    }, 
    
    onLoadGetEvents : function(){
        if (this.currentView === this.ViewType.calendar){
            this.onCalendarView(this.getDateViewElement());
        } else {
            this.onListView(this.getListViewElement());
        }
        this.hideTagHeader(true);
    },
    
    onChangePageSize : function (){
        var pageSize = this.getPageSize();
        //reset to first page when page size is changed; eventually try to calculate best match later...
        var pageNumber = 0;
        this.getEventsForList(pageSize, pageNumber);  
    },
    
    onListView : function (elem) {
        this.currentView = this.ViewType.list;
        //highlight active view
        this.setActiveView(elem);
        //display page size selection and footer (prev/next page buttons)
        this.hideMenu(false);
        this.toggleFooter();
        
        //reset search - start from beginning;
        var pageSize = this.getPageSize();
        var pageNumber = 0;
        
        setCookie(this.viewTypeCookieName, this.ViewType.list, null);
        this.getEventsForList(pageSize, pageNumber);
    }, 
    
    onCalendarView : function  (elem){
        this.currentView = this.ViewType.calendar;
        
        //highlight active view
        this.setActiveView(elem);
        //display calendar navigation and footer (download ICS button)
        this.hideMenu(false);
        this.toggleFooter();

        //TODO: reset search - start from beginning?What would be correct flow?
        //this.currentDate = new Date();
        
        var year = this.getCurrentYear();
        var month = this.getCurrentMonth();
        
        setCookie(this.viewTypeCookieName, this.ViewType.calendar, null);       
        this.getEventsForCalendar(year, month);
    },
    
    setActiveView : function (elem){
        var target  = CQ.Ext.get(elem);
        var anchorTagList = this.getEventListContainer().select(".eventlist-header a"); 

        //highlight label
        var activeClassName = "eventlist-view-active";
        anchorTagList.removeClass(activeClassName);
        target.addClass(activeClassName);
        
        activeClassName = "eventlist-view-active-" + this.getComponentName();
        anchorTagList.removeClass(activeClassName);
        target.addClass(activeClassName);
            
        //set active icon
        var activeListViewIconClassName = "eventlist-listview-icon-active";
        var activeDateViewIconClassName = "eventlist-dateview-icon-active";
        anchorTagList.removeClass(activeListViewIconClassName);
        anchorTagList.removeClass(activeDateViewIconClassName);
        if (this.currentView === this.ViewType.calendar){
            target.addClass(activeDateViewIconClassName);
        } else {
            target.addClass(activeListViewIconClassName);
        }
    },
    
    searchByTag : function (elem, tagId){
        this.currentSearchTag = tagId;
        //set "Back To Events" bar
        this.hideMenu(true);  
        this.hideTagHeader(false);
        
        var tagName = CQ.Ext.get('currentTag_'+ this.getComponentName());
        tagName.update(ellipsisString(this.tagsMap[tagId], 30));
        tagName.set({title: this.tagsMap[tagId] });
        this.tagActiveCss(elem, "eventlist-tag-active");
        
        if(this.currentView === this.ViewType.calendar) {
            //search within current month! 
            //Once "Back to Events" position for Calendar view is resolved this date might be reset
            var year = this.getCurrentYear();
            var month =this.getCurrentMonth();
            this.getEventsForCalendar(year, month);
        } else {
            var pageSize = this.getPageSize();
            var pageNumber = 0;
            this.getEventsForList(pageSize, pageNumber);
        }
        
        setAnchorParams("currentSearchTag", tagId);     
    },
    
    tagActiveCss : function (elem, activeClassName){
        var extElem  = CQ.Ext.get(elem);
        if ( null != extElem ) {
            this.removeActiveTagCss(extElem, activeClassName );
            //doesn't work for tag list when tag inside event summary is clicked;
            extElem.addClass(activeClassName);            
        }
    },

    removeActiveTagCss : function (extElem, activeClassName){
        var anchorTagList = this.getEventListContainer().select(".tag_list a");
        anchorTagList.removeClass(activeClassName);
    },
    
    hideMenu : function (hide){
        if (this.currentSearchTag == undefined){            
            if (this.currentView === this.ViewType.calendar) {
                hideElement('calendarNavigation_' + this.getComponentName(), hide);
                hideElement('pageSizeFilter_' + this.getComponentName(), !hide);
            } else {
                hideElement('pageSizeFilter_' + this.getComponentName(), hide);
                hideElement('calendarNavigation_' + this.getComponentName(), !hide);
            }
        } else {
            //if "search by tag" is active do not show menu
            hideElement('pageSizeFilter_' + this.getComponentName(), true);
            hideElement('calendarNavigation_' + this.getComponentName(), true );
        }
    },

    hideTagHeader : function (hide){
        hideElement('currentTagMessage_' + this.getComponentName(), hide);
        hideElement('backToEvents_' + this.getComponentName(), hide);
    },
    
    toggleFooter : function () {
        var hidePagination = false;
        if (this.currentView === this.ViewType.calendar) {
            hidePagination = true;
        }
        hideElement('pageNavigation_' + this.getComponentName(), hidePagination);
        hideElement('pageNo_' + this.getComponentName(), hidePagination);
        hideElement('calendarDownload_' + this.getComponentName(), !hidePagination);
    },

    backToEvents : function (elem){
        this.currentSearchTag = undefined;
        //remove highlight from tag
        var extElem  = CQ.Ext.get(elem);
        this.removeActiveTagCss(extElem, "eventlist-tag-active");
        
        this.onLoadGetEvents();
        window.location.href = getAnchorParams().url;
    }, 

    getEventsForList : function (pageSize, pageNumber){

         var params = {paramPageSize: pageSize, paramPageNumber: pageNumber};
         
         this.fillInfilterTags(params);

         this.loadEventItems (params);
    },

    getEventsForCalendar : function (year, month){
         var params = {paramYear: year, paramMonth: month};
       
         this.fillInfilterTags(params);

         this.loadEventItems (params);
    },
    
    fillInfilterTags : function(params) {
         //don't send empty tag list - API will not return any event.
         if(this.filterTagArray && this.filterTagArray.length>0){
             params['paramTagIdList'] = this.filterTagArray;
         }
        
         if(this.currentSearchTag){
             params['paramSubFilterTagId'] = this.currentSearchTag;
             //needed for search by tag message
             params['paramTagName'] = this.tagsMap[this.currentSearchTag];
         }
    },

    loadEventItems : function (params){
        
         var path = this.getComponentPath() + ".items.html" + this.pageParams;
         if (this.currentView === this.ViewType.calendar){
             path = this.getComponentPath() + ".calendar.html" + this.pageParams;
         }

        params['_charset_'] = "utf-8";
        
        var msgLoading = CQ.I18n.getMessage("eventListLoading");
        
        //cancel previous request if any exists
        this.getListPanel().getUpdater().abort();
                
        this.getListPanel().load({
             url:path,
             method:'GET',
             params: params,
             text: msgLoading,
             timeout: 30,
             nocache: true,
             scripts: true
        }); 
    },

    prev : function (){
        var prevPage = this.currentPage - 1;
        if(prevPage < 0){
            prevPage = 0;
        }
        this.getEventsForList(this.getPageSize(), prevPage);
    },

    next : function (){
        var nextPage = this.currentPage + 1;
        this.getEventsForList(this.getPageSize(), nextPage);
    },
    
    prevMonth : function (){
        this.currentDate = this.currentDate.add(Date.MONTH, -1);
        var year = this.getCurrentYear();
        var month = this.getCurrentMonth();
        this.getEventsForCalendar(year, month);
    },

    nextMonth : function (){
        this.currentDate = this.currentDate.add(Date.MONTH, 1);
        var year = this.getCurrentYear();
        var month = this.getCurrentMonth();
        this.getEventsForCalendar(year, month);
    }
}



